<?php

/**
 * The plugin bootstrap file
 *
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 *
 * @link              https://www.artisanweb.co.uk
 * @since             1.0.3
 * @package           Woocommerce_Trade_Account
 *
 * @wordpress-plugin
 * Plugin Name:       WooCommerce Trade Account
 * Plugin URI:        https://www.artisanweb.co.uk
 * Description:       This is a short description of what the plugin does. It's displayed in the WordPress admin area.
 * Version:           1.0.3
 * Author:            Daniel McFarland
 * Author URI:        https://www.artisanweb.co.uk
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       woocommerce-trade-account
 * Domain Path:       /languages
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Currently plugin version.
 * Start at version 1.0.3 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define( 'WOOCOMMERCE_TRADE_ACCOUNT_VERSION', '1.0.3' );


/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then kicking off the plugin from this point in the file does
 * not affect the page life cycle.
 *
 * @since    1.0.3
 */

 /**
  * Check if WooCommerce is active
  **/
 if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
	add_action( 'plugins_loaded', 'run_woocommerce_trade_account' );
}

//
// Function to disable Stripe Gateway
//

function bbloomer_paypal_disable_manager( $available_gateways ) {
	global $woocommerce;

	// Disable all Payment Gateways other than a Trade Account
	foreach ($available_gateways as $key => $current_gateway) {
		if($key != (new WC_Gateway_Your_Gateway)->get_id()) {
			unset( $available_gateways[$key] );
		}
	}

	return $available_gateways;
}

//
// Function to generate additional role for trade account
//

function aweb_trade_account_role() {

	// Trade Account role.
	add_role(
		'customer_trade',
		'Trade Account',
		array(
			'read' => true,
		)
	);
}

//
// Action to initiate Custom WC Gateway when plugins are loaded
//

function run_woocommerce_trade_account() {

	class WC_Gateway_Your_Gateway extends WC_Payment_Gateway {

		public function __construct() {
			$this->id = 'trade_account';
			$this->has_fields = true;
			$this->init_form_fields();
			$this->init_settings();

			// Actions
			add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
			add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );

		}

		public function field_name($field_name) {
			return str_replace('-', '_', $field_name);
		}

		public function payment_fields()
		{

			return $this->form();

		}

		/**
		 * Outputs fields for entering credit card information.
		 *
		 * @since 2.6.0
		 */
		public function form() {
			wp_enqueue_script( 'wc-credit-card-form' );
			$fields = array();

			$default_fields = array(
				'purchase-order-number-field' => '<p class="form-row form-row-wide">
					<label for="' . esc_attr( $this->id ) . '-purchase-order-number">' . esc_html__( 'Purchase Order Number', 'woocommerce' ) . '&nbsp;<span class="required">*</span></label>
					<input id="' . esc_attr( $this->id ) . '-purchase-order-number" name="' . $this->field_name( 'purchase-order-number' ) . '" class="input-text wc-credit-card-form-purchase-order-number"  autocorrect="no" autocapitalize="no" spellcheck="no" type="tel" placeholder="Enter the Purchase Order Number Here" ' . $this->field_name( 'purchase-order-number' ) . ' />
				</p>',
				'order-comments-field' => '<p class="form-row form-row-wide">
					<label for="' . esc_attr( $this->id ) . '-purchase-order-number">' . esc_html__( 'Additional Notes', 'woocommerce' ) . '</label>
					<textarea style="height: auto!important;" rows="4" id="' . esc_attr( $this->id ) . '-order-comments" name="' . $this->field_name( 'order-comments' ) . '" class="input-text wc-credit-card-form-order-comments" placeholder="Enter Additional Notes Here" ' . $this->field_name( 'order-comments' ) . ' ></textarea>
				</p>',
			);

			$fields = wp_parse_args( $fields, apply_filters( 'woocommerce_credit_card_form_fields', $default_fields, $this->id ) );
			?>

			<fieldset id="wc-<?php echo esc_attr( $this->id ); ?>-purchase-order-form" class='wc-purchase-order-form wc-payment-form'>
				<?php do_action( 'woocommerce_credit_card_form_start', $this->id ); ?>
				<?php
				foreach ( $fields as $field ) {
					echo $field; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
				}
				?>
				<?php do_action( 'woocommerce_credit_card_form_end', $this->id ); ?>
				<div class="clear"></div>
			</fieldset>
			<?php
		}

		/**
		 * Output for the order received page.
		 */
		public function thankyou_page($order) {
			// $order_id = $order->get_id();

			$order_id = $order;


			if(get_post_meta( $order_id, '_aweb_trade_notes', true ) != '') { ?>

				<h2 style="margin: 0 0 5px;">Additional Notes</h2>
				<table style="margin-bottom: 30px;">
					<tr>
						<td><?php echo wptexturize( get_post_meta( $order_id, '_aweb_trade_notes', true ) ); ?></td>
					</tr>
				</table>

			<?php }
		}

		public function has_fields() {
			return true;
		}

		public function get_id() {
			return $this->id;
		}

		public function get_method_title() {
			return 'WooCommerce Trade Account';
		}

		public function get_method_description() {
			return 'This allows specific users to use a trade account to pay for their order';
		}

		public function get_title() {
			return 'Trade Account';
		}

		public function allowSavedCards() {
			return false;
		}

		public function get_description() {
			return 'Pay for your order using a PO Number.';
		}

		public function is_available() {

			if(is_user_logged_in()) {

				$user = wp_get_current_user();
				if ( in_array( 'customer_trade', (array) $user->roles ) ) {

					// Disable Other Payment Gateways if Logged in and is Trade Account
					add_filter( 'woocommerce_available_payment_gateways', 'bbloomer_paypal_disable_manager' );

					// Set Trade Account As Only Payment Gateway
					$this->set_current();
					return true;
				}
			}

			// Disable Trade Account Gateway
			return false;
		}


		/**
		 * Process the payment and return the result
		 *
		 * @param int $order_id
		 * @return array
		 */
		public function process_payment( $order_id ) {

			$PO_number = $_POST['purchase_order_number'];
			$PO_notes = $_POST['order_comments'];

			if($PO_number == '') {
				wc_add_notice( 'You need to enter a PO Number', 'error' );
				return;
			}

			$current_user_id = get_current_user_id();

			$po_numbers = get_option( '_' . $current_user_id . '_po_numbers' );

			// var_dump($po_numbers);
			//
			// die();

			if($po_numbers == 'false') {
				$po_numbers = [];
			} else {
				$po_numbers = json_decode($po_numbers);
			}

			foreach ($po_numbers as $key => $po_number_) {
				if($po_number_ == $PO_number) {
					wc_add_notice( 'PO Number Has Already Been Used', 'error' );
					return;
				}
			}

			array_push($po_numbers, $PO_number);

			$po_numbers = json_encode($po_numbers);

			update_option( '_' . $current_user_id . '_po_numbers', $po_numbers );


			// check PO Number is valid
			if($is_error = false) {
				wc_add_notice( 'Issue with PO Number', 'error' );
				return;
			}

			// Get the current order
			$order = wc_get_order( $order_id );

			update_post_meta($order_id, '_aweb_trade_notes', $PO_notes);

			// Set Payment Method Title to PO Number
			$order->set_payment_method_title( 'Purchase Order: ' . $PO_number );
			$order->save();

			// Mark as payment complete
			$order->update_status( 'processing', 'Purchase Order: ' . $PO_number . ' | ' );
			// $order->payment_complete();

			// Add the note
			$order->add_order_note( 'Purchase Order: ' . $PO_number );

			// Remove cart
			WC()->cart->empty_cart();

			// Return thankyou redirect
			return array(
				'result' 	=> 'success',
				'redirect'	=> $this->get_return_url( $order )
			);
		}

	}

}
register_activation_hook( __FILE__, 'aweb_trade_account_role' );

//
//
//

function add_aweb_trade_account_gateway( $methods ) {
    $methods[] = 'WC_Gateway_Your_Gateway';
    return $methods;
}

add_filter( 'woocommerce_payment_gateways', 'add_aweb_trade_account_gateway' );


//
//
//

add_action( 'woocommerce_email_after_order_table', 'woocommerce_email_after_order_table_func' );
function woocommerce_email_after_order_table_func( $order ) {

	$order_id = $order->get_id();

	if(get_post_meta( $order_id, '_aweb_trade_notes', true ) != '') { ?>

		<h3 style="margin: 0 0 5px;">Additional Notes</h3>
		<table style="margin-bottom: 30px;">
			<tr>
				<td><?php echo wptexturize( get_post_meta( $order_id, '_aweb_trade_notes', true ) ); ?></td>
			</tr>
		</table>

	<?php }
}

//
//
//

// Display field value on the order edit page (not in custom fields metabox)
add_action( 'woocommerce_admin_order_data_after_billing_address', 'my_custom_checkout_field_display_admin_order_meta', 10, 1 );
function my_custom_checkout_field_display_admin_order_meta($order){
    $my_custom_field = get_post_meta( $order->get_id(), '_aweb_trade_notes', true );
    if ( ! empty( $my_custom_field ) ) {
        echo '<p><strong>Additional Notes:</strong> ' . get_post_meta( $order->get_id(), '_aweb_trade_notes', true ) . '</p>';
    }
}
