<?php

/**
 * The plugin bootstrap file
 *
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 *
 * @link              https://www.artisanweb.co.uk
 * @since             1.0.0
 * @package           Woocommerce_Wishlist
 *
 * @wordpress-plugin
 * Plugin Name:       WooCommerce Wishlist
 * Plugin URI:        https://www.artisanweb.co.uk
 * Description:       This is a short description of what the plugin does. It's displayed in the WordPress admin area.
 * Version:           1.0.0
 * Author:            Daniel McFarland
 * Author URI:        https://www.artisanweb.co.uk
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       woocommerce-wishlist
 * Domain Path:       /languages
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Currently plugin version.
 * Start at version 1.0.0 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define( 'WOOCOMMERCE_WISHLIST_VERSION', '1.0.0' );


/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then kicking off the plugin from this point in the file does
 * not affect the page life cycle.
 *
 * @since    1.0.0
 */

 /**
  * Check if WooCommerce is active
  **/
 if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
	add_action( 'plugins_loaded', 'run_woocommerce_wishlist' );
}

function run_woocommerce_wishlist() {

	add_action( 'woocommerce_wishlist_header_heart', 'woocommerce_wishlist_view_wishlist', 10, 0 );

	add_action( 'woocommerce_wishlist_add_remove', 'woocommerce_wishlist_toggle_item', 10, 1 );

	// add_action( 'template_redirect', 'woocommerce_wishlist_page' );

	// Hook my above function to the pre_get_posts action
	add_action( 'pre_get_posts', 'my_modify_main_query' );

	// Get the correct template for the wishlist
	add_filter('template_include', 'lessons_template', 100);

	// create custom plugin settings menu
	add_action('admin_menu', 'aweb_woocommerce_wishlist_admin');

	// Override default page title
	add_filter('document_title_parts', 'dq_override_post_title', 10);
	add_filter('wpseo_title', 'dq_override_post_title_wpseo');

}

function woocommerce_wishlist_view_wishlist() {

	if(is_user_logged_in()) {
		// get wishlist page url
		$wish_list_url = get_permalink( get_option('woocommerce_wishlist_page_id') );
	} else {
		$wish_list_url = get_permalink( get_option('woocommerce_myaccount_page_id') );
	} ?>

	<a href="<?= $wish_list_url; ?>">
		<img alt="wishlist icon" src="<?= get_stylesheet_directory_uri(); ?>/assets/images/heart.svg">
	</a>

<?php }

function woocommerce_wishlist_toggle_item($product_id) {

	if(is_user_logged_in()) {

		// Get Current User
		$current_user_id = get_current_user_id();

		// Get Array from DB
		$wishlist = get_option( '_' . $current_user_id . '_wishlist_items', false );

		if($wishlist == false || $wishlist == '') {
			$wishlist = [];
		} else {
			$wishlist = json_decode($wishlist);
		}

		// check if product is on wishlist
		if(in_array ( $product_id, $wishlist )) {
			// Remove from wishlist
			$wishlist_toggle_url = admin_url('admin-post.php') . '?action=remove_wishlist&product_id=' . $product_id;
			$on_wishlist = true;
		} else {
			// Add to wishlist
			$wishlist_toggle_url = admin_url('admin-post.php') . '?action=add_wishlist&product_id=' . $product_id;
			$on_wishlist = false;
		}

	} else {
		$wishlist_toggle_url = get_permalink( get_option('woocommerce_myaccount_page_id') );

		$on_wishlist = false;

	} ?>

	<div class="column small-6 large-6 single--product-share-column">
		<a href="<?= $wishlist_toggle_url; ?>" class="row no-split align-middle">
			<div class="column small-4 large-3 single--product-share-icon">
				<img src="<?php echo get_stylesheet_directory_uri(); ?>/assets/images/heart.svg">
			</div>
			<div class="column small-8 large-9 single--product-share-text">
				<?php if($on_wishlist) { ?>
					<h3>Remove From Wishlist</h3>
				<?php } else { ?>
					<h3>Save To Your Wishlist</h3>
				<?php } ?>
			</div>
		</a>
	</div>


<?php }

//
//
//

add_action( 'admin_post_add_wishlist', 'woocommerce_wishlist_add' );

function woocommerce_wishlist_add() {

	// Store Product ID to Add to Wishlist
	$product_id = $_REQUEST['product_id'];
	$current_user_id = get_current_user_id();

	// Get Array from DB
	$wishlist = get_option( '_' . $current_user_id . '_wishlist_items', false );

	if($wishlist == false || $wishlist == '') {
		$wishlist = [];
	} else {
		$wishlist = json_decode($wishlist);
	}

	// Add to array
	array_push($wishlist, $product_id);

	// Clean Array
	$wishlist = array_values(array_unique($wishlist));

	// Encode Array in JSON for DB
	$wishlist = json_encode($wishlist);

	// Save Wishlist
	update_option( '_' . $current_user_id . '_wishlist_items', $wishlist );

	// Redirect to Product Page
	wp_redirect(get_permalink($product_id));
}

add_action( 'admin_post_remove_wishlist', 'woocommerce_wishlist_remove' );

function woocommerce_wishlist_remove() {
    // Handle request then generate response using echo or leaving PHP and using HTML

	// Store Product ID to Remove from Wishlist
	$product_id = $_REQUEST['product_id'];
	$current_user_id = get_current_user_id();

	// Get Array from DB
	$wishlist = get_option( '_' . $current_user_id . '_wishlist_items', false );
	if($wishlist == false || $wishlist == '') {
		$wishlist = [];
	} else {
		$wishlist = json_decode($wishlist);
	}

	// Find and delete item in Wishlist
	if (($key = array_search($product_id, $wishlist)) !== false) {
		unset($wishlist[$key]);
	}

	// Clean Array
	$wishlist = array_values(array_unique($wishlist));

	// Encode Array in JSON for DB
	$wishlist = json_encode($wishlist);

	// Save Wishlist
	update_option( '_' . $current_user_id . '_wishlist_items', $wishlist );

	// Redirect to Product Page
	wp_redirect(get_permalink($product_id));
}

//
//
//

// My function to modify the main query object
function my_modify_main_query( $query ) {

	$wishlist_page = get_option( 'woocommerce_wishlist_page_id', 0 );

	// if not wishlist page return normal content
	if($query->queried_object_id != $wishlist_page) {
		return;
	}

	// get current user id
	$current_user_id = get_current_user_id();

	// Get Array from DB
	$wishlist = get_option( '_' . $current_user_id . '_wishlist_items', false );

	if($wishlist == false || $wishlist == '') {
		$wishlist = [];
	} else {
		$wishlist = json_decode($wishlist);
	}

	// query products from wishlist
	$args = array(
	   'post__in' => $wishlist,
	   'post_type' => 'product',
	   'posts_per_page' => -1,
	);

	global $wp_query;

	$query = new WP_Query( $args );

	$wp_query = $query;

	// set up query to be wishlist
	set_query_var('is_wishlist', true);
	set_query_var('is_archive', true);
	wc_set_loop_prop('total', count($wishlist));

}

function lessons_template( $template ) {

  if ( get_query_var('is_wishlist') ) {
	  $template = plugin_dir_path(__FILE__) . 'woocommerce-wishlist-template.php';
  }

  return $template;
}

function dq_override_post_title($title){

	if(get_query_var('is_wishlist')) {
		$title['title'] = 'My Wishlist';
	}

    return $title;

};

function dq_override_post_title_wpseo($title){
	if(get_query_var('is_wishlist')) {
		return;
	}

	return $title;
}

//
// Admin Options
//

function aweb_woocommerce_wishlist_admin() {

	//create new top-level menu
	// add_menu_page('My Cool Plugin Settings', 'Cool Settings', 'administrator', __FILE__, 'aweb_woocommerce_wishlist_admin_page' , plugins_url('/images/icon.png', __FILE__) );

	add_submenu_page( 'woocommerce', 'Wishlist', 'Wishlist', 'manage_options', 'aweb-wishlist', 'aweb_woocommerce_wishlist_admin_page' );

	// add_options_page('My First WordPress Plugin Options', 'My First Plugin', 'manage_options', 'mfwp-options', 'aweb_woocommerce_wishlist_admin_page');

	//call register settings function
	add_action( 'admin_init', 'register_aweb_woocommerce_wishlist_settings' );
}


function register_aweb_woocommerce_wishlist_settings() {
	//register our settings
	// register_setting( 'aweb-woocommerce-wishlist-group', 'new_option_name' );
	// register_setting( 'aweb-woocommerce-wishlist-group', 'some_other_option' );
	// register_setting( 'aweb-woocommerce-wishlist-group', 'option_etc' );
	register_setting( 'aweb-woocommerce-wishlist-group', 'woocommerce_wishlist_page_id');
}

function aweb_woocommerce_wishlist_admin_page() {
?>
<div class="wrap">
<h1>WooCommerce Wishlist (v<?= WOOCOMMERCE_WISHLIST_VERSION; ?>)</h1>

<form method="post" action="options.php">
    <?php
	settings_fields( 'aweb-woocommerce-wishlist-group' );
	do_settings_sections( 'aweb-woocommerce-wishlist-group' );
	$pages = get_pages();
	?>
	<table class="form-table">
		<!-- <tr valign="top">
			<th scope="row">New Option Name</th>
			<td><input type="text" name="new_option_name" value="<?php echo esc_attr( get_option('new_option_name') ); ?>" /></td>
		</tr>

		<tr valign="top">
			<th scope="row">Some Other Option</th>
			<td><input type="text" name="some_other_option" value="<?php echo esc_attr( get_option('some_other_option') ); ?>" /></td>
		</tr> -->

		<tr valign="top">
			<th scope="row">Page to Display Wishlist</th>
			<td>
				<select name="woocommerce_wishlist_page_id" id="woocommerce_wishlist_page_id">
					<?php foreach($pages as $page) { ?>
						<?php if(esc_attr( get_option('woocommerce_wishlist_page_id') ) == $page->ID) { $selected = 'selected="selected"'; } else { $selected = ''; } ?>
						<option value="<?php echo $page->ID; ?>" <?php echo $selected; ?>><?php echo $page->post_title; ?></option>
					<?php } ?>
				</select>
			</td>
		</tr>
	</table>

    <?php submit_button(); ?>

</form>
</div>
<?php }
