<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Paypal Pro Hosted Gateway.
 */
class WC_Gateway_Paypal_Pro_Hosted extends WC_Payment_Gateway {

	/**
	 * Init and hook in the gateway.
	 */
	public function __construct() {
		$this->id                 = 'paypal-pro-hosted';
		$this->has_fields         = false;
		$this->method_title       = __( 'PayPal Pro Hosted', 'woocommerce-gateway-paypal-pro-hosted' );
		$this->method_description = __( 'Accept credit and debit card or PayPal payments without capturing or storing card information on your website.', 'woocommerce-gateway-paypal-pro-hosted' );
		$this->order_button_text  = __( 'Proceed to payment', 'woocommerce-gateway-paypal-pro-hosted' );
		$this->supports           = array(
			'products',
			'refunds'
		);

		// Load the form fields.
		$this->init_form_fields();

		// Load the settings.
		$this->init_settings();

		// Define user set variables.
		$this->title            = $this->get_option( 'title' );
		$this->description      = $this->get_option( 'description' );
		$this->receiver_email   = $this->get_option( 'receiver_email' );
		$this->api_username     = $this->get_option( 'api_username' );
		$this->api_password     = $this->get_option( 'api_password' );
		$this->api_signature    = $this->get_option( 'api_signature' );
		$this->invoice_prefix   = $this->get_option( 'invoice_prefix', 'WC-' );
		$this->payment_action   = $this->get_option( 'payment_action' );
		$this->sandbox          = $this->get_option( 'sandbox' );
		$this->debug            = $this->get_option( 'debug' );

		// Active logs.
		if ( 'yes' == $this->debug ) {
			$this->log = new WC_Logger();
		}

		// Set the API.
		$this->api = new WC_Paypal_Pro_Hosted_API( $this );

		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );
		add_action( 'woocommerce_wc_paypal_hosted_pro_ipn_request', array( $this, 'order_status_change' ) );
		add_action( 'woocommerce_api_wc_gateway_paypal_pro_hosted', array( $this, 'check_ipn_response' ) );

		$this->admin_notices();
	}

	/**
	 * Test the API credentials when settings are saved.
	 *
	 * @since 1.0.22
	 * @return void
	 */
	public function process_admin_options() {
		parent::process_admin_options();

		// test the API Credentials each time the settings are saved
		if ( ! empty( $this->get_option( 'api_username' ) ) && ! empty( $this->get_option( 'api_password' ) ) && ! empty( $this->get_option( 'api_signature' ) ) ) {
			$valid_api_credentials = $this->api->test_api_credentials();

			if ( ! $valid_api_credentials ) {
				WC_Admin_Settings::add_error( __( 'Error: The API credentials you provided are invalid. Please confirm that your PayPal account has been verified and you have entered the correct information before trying again.', 'woocommerce-gateway-paypal-pro-hosted' ) );
			}
		}
	}

	/**
	 * Get supported currencies.
	 *
	 * @return array List of supported currencies.
	 */
	public function get_supported_currencies() {
		return apply_filters( 'woocommerce_paypal_pro_hosted_supported_currencies', array( 'AUD', 'CAD', 'CHF', 'CZK', 'DKK', 'EUR', 'GBP', 'HKD', 'HUF', 'JPY', 'NOK', 'NZD', 'PLN', 'SEK', 'SGD', 'USD' ) );
	}

	/**
	 * Get supported countries.
	 *
	 * @return array List of supported countries.
	 */
	public function get_supported_countries() {
		return apply_filters( 'woocommerce_paypal_pro_hosted_supported_countries', array( 'GB', 'FR', 'AU', 'HK', 'IT', 'ES', 'JP' ) );
	}

	/**
	 * Returns a value indicating the the Gateway is available or not. It's called
	 * automatically by WooCommerce before allowing customers to use the gateway
	 * for payment.
	 *
	 * @return bool
	 */
	public function is_available() {
		// Test if is valid for use.
		$available = parent::is_available() &&
					! empty( $this->receiver_email ) &&
					! empty( $this->api_username ) &&
					! empty( $this->api_password ) &&
					! empty( $this->api_signature ) &&
					in_array( get_woocommerce_currency(), $this->get_supported_currencies() );

		return $available;
	}

	/**
	 * Displays notifications when the admin has something wrong with the configuration.
	 */
	protected function admin_notices() {
		if ( is_admin() && 'yes' == $this->get_option( 'enabled' ) ) {
			// Checks if email is not empty.
			if ( empty( $this->receiver_email ) || empty( $this->api_username ) || empty( $this->api_password ) || empty( $this->api_signature ) ) {
				add_action( 'admin_notices', array( $this, 'plugin_not_configured_message' ) );
			} else if ( ! $this->api_credentials_verified() ) {
				add_action( 'admin_notices', array( $this, 'api_credentials_not_valid_message' ) );
			}

			// Checks that the currency is supported.
			if ( ! in_array( get_woocommerce_currency(), $this->get_supported_currencies() ) ) {
				add_action( 'admin_notices', array( $this, 'currency_not_supported_message' ) );
			}

			// Checks that the country is supported.
			if ( ! in_array( WC()->countries->get_base_country(), $this->get_supported_countries() ) ) {
				add_action( 'admin_notices', array( $this, 'country_not_supported_message' ) );
			}
		}
	}

	/**
	 * Checks if the current API credentials set have been tested/verified and display
	 * an admin notice if the API credentials are invalid.
	 *
	 * If the API creds haven't been tested yet, test them first before displaying an admin if they're invalid
	 *
	 * @since 1.0.22
	 * @return bool
	 */
	public function api_credentials_verified() {
		$credentials_verified = get_transient( 'wc_paypal_pro_hosted_' . ( 'yes' == $this->sandbox ? 'sandbox_' : '' ) . 'account_' . md5( $this->api_username . ':' . $this->api_password . ':' . $this->api_signature ) . '_verified' );

		return empty( $credentials_verified ) ? $this->api->test_api_credentials() : ( ( 'yes' === $credentials_verified ) ? true : false );
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {

		if ( function_exists( 'wc_get_log_file_path' ) ) {
			$log_path = '<code>' . wc_get_log_file_path( $this->id ) . '</code>';
		} else {
			$log_path = '<code>woocommerce/logs/' . esc_attr( $this->id ) . '-%s.txt</code>';
		}

		$this->form_fields = array(
			'enabled' => array(
				'title'   => __( 'Enable/Disable', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable PayPal Pro Hosted', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default' => 'yes'
			),
			'title' => array(
				'title'       => __( 'Title', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => __( 'Credit Card', 'woocommerce-gateway-paypal-pro-hosted' ),
				'desc_tip'    => true,
			),
			'description' => array(
				'title'       => __( 'Description', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'textarea',
				'description' => __( 'This controls the description which the user sees during checkout.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => __( 'Pay with credit card', 'woocommerce-gateway-paypal-pro-hosted' )
			),
			'receiver_email' => array(
				'title'       => __( 'Receiver Email', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'email',
				'description' => __( 'Input your PayPal receiver email account.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => '',
				'desc_tip'    => true,
				'placeholder' => 'you@youremail.com'
			),
			'api_credentials' => array(
				'title'       => __( 'API Credentials', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'title',
				'description' => '',
			),
			'api_username' => array(
				'title'       => __( 'Username', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'text',
				'description' => __( 'Please enter your PayPal API username; this is needed in order to take payment.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'api_password' => array(
				'title'       => __( 'Password', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'text',
				'description' => __( 'Please enter your PayPal API password; this is needed in order to take payment.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'api_signature' => array(
				'title'       => __( 'Signature', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'text',
				'description' => __( 'Please enter your PayPal API signature; this is needed in order to take payment.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'shipping' => array(
				'title'       => __( 'Shipping options', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'title',
				'description' => '',
			),
			'advanced' => array(
				'title'       => __( 'Advanced options', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'title',
				'description' => ''
			),
			'invoice_prefix' => array(
				'title'       => __( 'Invoice Prefix', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'text',
				'description' => __( 'Please enter a prefix for your invoice numbers. If you use your PayPal account for multiple stores ensure this prefix is unqiue as PayPal will not allow orders with the same invoice number.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'desc_tip'    => true,
				'default'     => 'WC-'
			),
			'payment_action' => array(
				'title'       => __( 'Payment Action', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'select',
				'description' => __( 'Choose whether you wish to capture funds immediately or authorize payment only.', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => 'sale',
				'desc_tip'    => true,
				'options'     => array(
					'sale'          => __( 'Capture', 'woocommerce-gateway-paypal-pro-hosted' ),
					'authorization' => __( 'Authorize', 'woocommerce-gateway-paypal-pro-hosted' )
				)
			),
			'testing' => array(
				'title'       => __( 'Gateway Testing', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'title',
				'description' => '',
			),
			'sandbox' => array(
				'title'       => __( 'Sandbox', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable PayPal sandbox', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => 'no',
				'description' => sprintf( __( 'PayPal sandbox can be used to test payments. Sign up for a developer account %s.', 'woocommerce-gateway-paypal-pro-hosted' ), '<a href="https://developer.paypal.com/webapps/developer/applications/accounts" target="_blank">' . __( 'here', 'woocommerce-gateway-paypal-pro-hosted' ) . '</a>' ),
			),
			'debug' => array(
				'title'       => __( 'Debug Log', 'woocommerce-gateway-paypal-pro-hosted' ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable logging', 'woocommerce-gateway-paypal-pro-hosted' ),
				'default'     => 'no',
				'description' => sprintf( __( 'Log PayPal Pro Hosted events, inside %s', 'woocommerce-gateway-paypal-pro-hosted' ), $log_path ),
			)
		);
	}

	/**
	 * Get icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		$icon_html = '';
		$icon      = (array) $this->get_icon_image( WC()->countries->get_base_country() );

		foreach ( $icon as $i ) {
			$icon_html .= '<img src="' . esc_attr( $i ) . '" alt="' . __( 'PayPal Acceptance Mark', 'woocommerce-gateway-paypal-pro-hosted' ) . '" />';
		}

		$icon_html .= sprintf( '<a style="float: right; line-height: 52px; font-size: .83em;" href="%1$s" class="about_paypal" onclick="javascript:window.open(\'%1$s\',\'WIPaypal\',\'toolbar=no, location=no, directories=no, status=no, menubar=no, scrollbars=yes, resizable=yes, width=1060, height=700\'); return false;" title="' . esc_attr__( 'What is PayPal?', 'woocommerce-gateway-paypal-pro-hosted' ) . '">' . esc_attr__( 'What is PayPal?', 'woocommerce-gateway-paypal-pro-hosted' ) . '</a>', esc_url( $this->get_icon_url( WC()->countries->get_base_country() ) ) );

		return apply_filters( 'woocommerce_gateway_icon', $icon_html, $this->id );
	}

	/**
	 * Get the link for an icon based on country
	 *
	 * @param  string $country
	 *
	 * @return string
	 */
	private function get_icon_url( $country ) {
		switch ( $country ) {
			case 'MX' :
				$link = 'https://www.paypal.com/mx/cgi-bin/webscr?cmd=xpt/Marketing/general/WIPaypal-outside';
			break;
			default :
				$link = 'https://www.paypal.com/' . strtolower( $country ) . '/webapps/mpp/paypal-popup';
			break;
		}

		return $link;
	}

	/**
	 * Get PayPal images for a country.
	 *
	 * @param  string $country
	 *
	 * @return array of image URLs
	 */
	private function get_icon_image( $country ) {
		switch ( $country ) {
			case 'US' :
			case 'NZ' :
			case 'CZ' :
			case 'HU' :
			case 'MY' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/logo/AM_mc_vs_dc_ae.jpg';
			break;
			case 'TR' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/logo-center/logo_paypal_odeme_secenekleri.jpg';
			break;
			case 'GB' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/Logo/AM_mc_vs_ms_ae_UK.png';
			break;
			case 'MX' :
				$icon = array(
					'https://www.paypal.com/es_XC/Marketing/i/banner/paypal_visa_mastercard_amex.png',
					'https://www.paypal.com/es_XC/Marketing/i/banner/paypal_debit_card_275x60.gif'
				);
			break;
			case 'FR' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/logo-center/logo_paypal_moyens_paiement_fr.jpg';
			break;
			case 'AU' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_AU/mktg/logo/Solutions-graphics-1-184x80.jpg';
			break;
			case 'DK' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/logo-center/logo_PayPal_betalingsmuligheder_dk.jpg';
			break;
			case 'RU' :
				$icon = 'https://www.paypalobjects.com/webstatic/ru_RU/mktg/business/pages/logo-center/AM_mc_vs_dc_ae.jpg';
			break;
			case 'NO' :
				$icon = 'https://www.paypalobjects.com/webstatic/mktg/logo-center/banner_pl_just_pp_319x110.jpg';
			break;
			case 'CA' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_CA/mktg/logo-image/AM_mc_vs_dc_ae.jpg';
			break;
			case 'HK' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_HK/mktg/logo/AM_mc_vs_dc_ae.jpg';
			break;
			case 'SG' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_SG/mktg/Logos/AM_mc_vs_dc_ae.jpg';
			break;
			case 'TW' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_TW/mktg/logos/AM_mc_vs_dc_ae.jpg';
			break;
			case 'TH' :
				$icon = 'https://www.paypalobjects.com/webstatic/en_TH/mktg/Logos/AM_mc_vs_dc_ae.jpg';
			break;
			default :
				$icon = WC_HTTPS::force_https_url( plugins_url( 'assets/images/paypal.png', plugin_dir_path( __FILE__ ) ) );
			break;
		}

		return apply_filters( 'woocommerce_paypal_icon', $icon );
	}

	/**
	 * Get the transaction URL.
	 *
	 * @param WC_Order $order Order object.
	 *
	 * @version 1.10.11
	 *
	 * @return string
	 */
	public function get_transaction_url( $order ) {
		if ( 'yes' === $this->sandbox ) {
			$this->view_transaction_url = 'https://www.sandbox.paypal.com/cgi-bin/webscr?cmd=_view-a-trans&id=%s';
		} else {
			$this->view_transaction_url = 'https://www.paypal.com/cgi-bin/webscr?cmd=_view-a-trans&id=%s';
		}

		return parent::get_transaction_url( $order );
	}

	/**
	 * Process the payment and return the result.
	 *
	 * @param int    $order_id Order ID.
	 *
	 * @return array           Redirect.
	 */
	public function process_payment( $order_id ) {
		$order           = new WC_Order( $order_id );
		$button_token_id = $this->api->get_button_token_id( $order );

		if ( $button_token_id ) {
			update_post_meta( $order_id, '_paypal_pro_hosted_button', $button_token_id );

			return array(
				'result' 	=> 'success',
				'redirect'	=> $order->get_checkout_payment_url( true )
			);
		} else {
			wc_add_notice( '<strong>' . $this->title . '</strong>: ' . __( 'An error has occurred while processing your payment, please try again. Or contact us for assistance.', 'woocommerce-gateway-paypal-pro-hosted' ), 'error' );

			return array(
				'result'   => 'fail',
				'redirect' => ''
			);
		}
	}

	/**
	 * Output for the order received page.
	 *
	 * @param  $order_id Order ID.
	 *
	 * @return string    PayPal Pro Hosted iframe.
	 */
	public function receipt_page( $order_id ) {
		$order           = new WC_Order( $order_id );
		$button_token_id = get_post_meta( $order_id, '_paypal_pro_hosted_button', true );

		wc_enqueue_js( '
			$.blockUI({
				message: "' . esc_js( __( 'Please wait a few seconds to be able to make payment.', 'woocommerce-gateway-paypal-pro-hosted' ) ) . '",
				baseZ: 99999,
				overlayCSS: {
					background: "#fff",
					opacity:     0.6
				},
				css: {
					padding:         "20px",
					zindex:          "9999999",
					textAlign:       "center",
					color:           "#555",
					border:          "3px solid #aaa",
					backgroundColor: "#fff",
					cursor:          "wait",
					lineHeight:      "24px",
				}
			});

			$( "#paypal-pro-hosted-pay" ).hide();

			$( "#paypal-pro-hosted-iframe" ).on( "load", function () {
				$.unblockUI();
			});
		' );

		$html = '<iframe id="paypal-pro-hosted-iframe" src="' . $this->api->get_payment_url( $button_token_id ) . '" width="570" height="550" frameborder="0" style="width: 100%; max-width: 570px; margin: 0 auto; display: block; border: none !important;"></iframe>';

		$html .= '<div class="payment-buttons">';

			$html .= '<input type="submit" class="button alt" id="paypal-pro-hosted-pay" value="' . __( 'Pay via PayPal', 'woocommerce-gateway-paypal-pro-hosted' ) . '" />';
			$html .= '<a class="button cancel" href="' . esc_url( $order->get_cancel_order_url() ) . '">' . __( 'Cancel order &amp; restore cart', 'woocommerce-gateway-paypal-pro-hosted' ) . '</a>';

		$html .= '</div>';

		echo $html;
	}

	/**
	 * Send a notification to the user handling orders.
	 *
	 * @param  string $subject
	 * @param  string $message
	 */
	public function send_ipn_email_notification( $subject, $message ) {
		$new_order_settings = get_option( 'woocommerce_new_order_settings', array() );
		$mailer             = WC()->mailer();
		$message            = $mailer->wrap_message( $subject, $message );
		$to                 = ! empty( $new_order_settings['recipient'] ) ? $new_order_settings['recipient'] : get_option( 'admin_email' );

		$mailer->send( $to, $subject, $message );
	}

	/**
	 * Check for PayPal IPN Response.
	 */
	public function check_ipn_response() {

		@ob_clean();

		$ipn_response = ! empty( $_POST ) ? $_POST : false;

		if ( 'yes' == $this->debug ) {
			$this->log->add( $this->id, 'Found order #' . print_r( $ipn_response, true ) );
		}

		if ( $ipn_response && $this->api->check_ipn_request_is_valid( $ipn_response ) ) {

			header( 'HTTP/1.1 200 OK' );

			do_action( 'woocommerce_wc_paypal_hosted_pro_ipn_request', $ipn_response );

		} else {

			wp_die( __( 'PayPal IPN Request Failure', 'woocommerce-gateway-paypal-pro-hosted' ), __( 'PayPal IPN', 'woocommerce-gateway-paypal-pro-hosted' ), array( 'response' => 200 ) );

		}
	}

	/**
	 * Order status change handler.
	 *
	 * @param array $posted
	 */
	public function order_status_change( $posted ) {

		$posted = stripslashes_deep( $posted );

		// Custom holds post ID.
		if ( ! empty( $posted['invoice'] ) && ! empty( $posted['custom'] ) ) {

			$order = $this->api->get_order( $posted['custom'], $posted['invoice'] );
			$order_id = version_compare( WC_VERSION, '3.0', '<' ) ? $order->id : $order->get_id();

			if ( 'yes' == $this->debug ) {
				$this->log->add( $this->id, 'Found order #' . $order_id );
			}

			// Lowercase returned variables.
			$posted['payment_status'] = strtolower( $posted['payment_status'] );

			// txn_type may not be set, for example, in the case of a refund.
			if ( isset( $posted['txn_type'] ) ) {
				$posted['txn_type'] = strtolower( $posted['txn_type'] );
			}

			// Sandbox fix.
			if ( 1 == $posted['test_ipn'] && 'pending' == $posted['payment_status'] ) {
				$posted['payment_status'] = 'completed';
			}

			if ( 'yes' == $this->debug ) {
				$this->log->add( $this->id, 'Payment status: ' . $posted['payment_status'] );
			}

			$transaction_id =
				! empty( $posted['parent_txn_id'] ) ? wc_clean( $posted['parent_txn_id'] ) :
				( ! empty( $posted['txn_id'] ) ? wc_clean( $posted['txn_id'] ) : '' );

			// We are here so lets check status and do actions.
			switch ( $posted['payment_status'] ) {
				case 'completed' :
				case 'pending' :

					// Check order not already completed.
					if ( method_exists( $order, 'has_status' ) && $order->has_status( 'completed' ) || 'completed' === $order->get_status() ) {
						if ( 'yes' == $this->debug ) {
							$this->log->add( $this->id, 'Aborting, Order #' . $order_id . ' is already complete.' );
						}
						exit;
					}

					// Check valid txn_type.
					$accepted_types = array( 'cart', 'instant', 'express_checkout', 'web_accept', 'masspay', 'send_money', 'pro_hosted', 'paypal_integral_evolution' );

					if ( ! in_array( $posted['txn_type'], $accepted_types ) ) {
						if ( 'yes' == $this->debug ) {
							$this->log->add( $this->id, 'Aborting, Invalid type:' . $posted['txn_type'] );
						}
						exit;
					}

					// Validate currency.
					$order_currency = version_compare( WC_VERSION, '3.0', '<' ) ? $order->get_order_currency() : $order->get_currency();
					if ( $order_currency != $posted['mc_currency'] ) {
						if ( 'yes' == $this->debug ) {
							$this->log->add( $this->id, 'Payment error: Currencies do not match (sent "' . $order_currency . '" | returned "' . $posted['mc_currency'] . '")' );
						}

						// Put this order on-hold for manual checking.
						$order->update_status( 'on-hold', sprintf( __( 'Validation error: PayPal currencies do not match (code %s).', 'woocommerce-gateway-paypal-pro-hosted' ), $posted['mc_currency'] ) );
						exit;
					}

					// Validate amount.
					if ( $this->api->price_format( $order->get_total(), $order ) != $this->api->price_format( $posted['mc_gross'], $order ) ) {
						if ( 'yes' == $this->debug ) {
							$this->log->add( $this->id, 'Payment error: Amounts do not match (gross ' . $posted['mc_gross'] . ')' );
						}

						// Put this order on-hold for manual checking.
						$order->update_status( 'on-hold', sprintf( __( 'Validation error: PayPal amounts do not match (gross %s).', 'woocommerce-gateway-paypal-pro-hosted' ), $posted['mc_gross'] ) );
						exit;
					}

					// Validate Email Address.
					if ( strcasecmp( trim( $posted['receiver_email'] ), trim( $this->receiver_email ) ) != 0 ) {
						if ( 'yes' == $this->debug ) {
							$this->log->add( $this->id, 'IPN Response is for another one: ' . $posted['receiver_email'] . ' our email is ' . $this->receiver_email );
						}

						// Put this order on-hold for manual checking.
						$order->update_status( 'on-hold', sprintf( __( 'Validation error: PayPal IPN response from a different email address (%s).', 'woocommerce-gateway-paypal-pro-hosted' ), $posted['receiver_email'] ) );

						exit;
					}

					// Store PP Details.
					if ( ! empty( $posted['payer_email'] ) ) {
						update_post_meta( $order_id, 'Payer PayPal address', wc_clean( $posted['payer_email'] ) );
					}
					if ( empty( $posted['first_name'] ) && ! empty( $posted['address_name'] ) ) {
						update_post_meta( $order_id, 'Payer name', wc_clean( $posted['address_name'] ) );
					}
					if ( ! empty( $posted['first_name'] ) ) {
						update_post_meta( $order_id, 'Payer first name', wc_clean( $posted['first_name'] ) );
					}
					if ( ! empty( $posted['last_name'] ) ) {
						update_post_meta( $order_id, 'Payer last name', wc_clean( $posted['last_name'] ) );
					}
					if ( ! empty( $posted['payment_type'] ) ) {
						update_post_meta( $order_id, 'Payment type', wc_clean( $posted['payment_type'] ) );
					}

					if ( $posted['payment_status'] == 'completed' ) {
						if ( 'sale' == $this->payment_action ) {
							$payment_message = __( 'Payment captured.', 'woocommerce-gateway-paypal-pro-hosted' );
						} else {
							$payment_message = __( 'Payment authorized (you must capture this value).', 'woocommerce-gateway-paypal-pro-hosted' );
						}

						$order->add_order_note( sprintf( __( 'IPN payment completed. %s', 'woocommerce-gateway-paypal-pro-hosted' ), $payment_message ) );

						$order->payment_complete( $transaction_id );
					} else {
						$order->update_status( 'on-hold', sprintf( __( 'Payment pending: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $posted['pending_reason'] ) );
					}

					if ( 'yes' == $this->debug ) {
						$this->log->add( $this->id, 'Payment complete.' );
					}

				break;
				case 'denied' :
				case 'expired' :
				case 'failed' :
				case 'voided' :
					// Order failed
					$order->update_status( 'failed', sprintf( __( 'Payment %s via IPN.', 'woocommerce-gateway-paypal-pro-hosted' ), strtolower( $posted['payment_status'] ) ) );
				break;
				case 'refunded' :

					// Only handle full refunds, not partial
					if ( $order->get_total() == ( $posted['mc_gross'] * -1 ) ) {

						// Mark order as refunded
						$order->update_status( 'refunded', sprintf( __( 'Payment %s via IPN.', 'woocommerce-gateway-paypal-pro-hosted' ), strtolower( $posted['payment_status'] ) ) );

						$this->send_ipn_email_notification(
							sprintf( __( 'Payment for order %s refunded/reversed', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number() ),
							sprintf( __( 'Order %s has been marked as refunded - PayPal reason code: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number(), $posted['reason_code'] )
						);
					}

				break;
				case 'reversed' :

					// Mark order as refunded
					$order->update_status( 'on-hold', sprintf( __( 'Payment %s via IPN.', 'woocommerce-gateway-paypal-pro-hosted' ), strtolower( $posted['payment_status'] ) ) );

					$this->send_ipn_email_notification(
						sprintf( __( 'Payment for order %s reversed', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number() ),
						sprintf(__( 'Order %s has been marked on-hold due to a reversal - PayPal reason code: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number(), $posted['reason_code'] )
					);

				break;
				case 'canceled_reversal' :
					$this->send_ipn_email_notification(
						sprintf( __( 'Reversal cancelled for order %s', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number() ),
						sprintf( __( 'Order %s has had a reversal cancelled. Please check the status of payment and update the order status accordingly.', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number() )
					);
				break;
				default :
					// No action
				break;
			}

			if ( ! empty( $transaction_id ) ) {
				update_post_meta( $order_id, '_transaction_id', $transaction_id );
			}

			exit;
		}
	}

	/**
	 * Process a refund if supported.
	 *
	 * @param  int    $order_id
	 * @param  float  $amount
	 * @param  string $reason
	 *
	 * @return bool|WP_Error True or false based on success, or a WP_Error object.
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		$order = new WC_Order( $order_id );

		if ( ! $order || ! $order->get_transaction_id() || ! $this->api_username || ! $this->api_password || ! $this->api_signature ) {
			return false;
		}

		$response = $this->api->do_refund( $order, $amount, $reason );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( empty( $response['body'] ) ) {
			return new WP_Error( 'paypal-error', __( 'Empty Paypal response.', 'woocommerce-gateway-paypal-pro-hosted' ) );
		}

		parse_str( $response['body'], $parsed_response );

		switch ( strtolower( $parsed_response['ACK'] ) ) {
			case 'success':
			case 'successwithwarning':
				$order->add_order_note( sprintf( __( 'Refunded %s - Refund ID: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $parsed_response['GROSSREFUNDAMT'], $parsed_response['REFUNDTRANSACTIONID'] ) );
				return true;
			break;
		}

		return false;
	}

	/**
	 * Adds error message when the plugin is not configured properly.
	 *
	 * @return string
	 */
	public function plugin_not_configured_message() {
		$id = 'woocommerce_' . $this->id . '_';
		if (
			isset( $_POST[ $id . 'receiver_email' ] ) && ! empty( $_POST[ $id . 'receiver_email' ] )
			&& isset( $_POST[ $id . 'api_username' ] ) && ! empty( $_POST[ $id . 'api_username' ] )
			&& isset( $_POST[ $id . 'api_password' ] ) && ! empty( $_POST[ $id . 'api_password' ] )
			&& isset( $_POST[ $id . 'api_signature' ] ) && ! empty( $_POST[ $id . 'api_signature' ] )
		) {
			return;
		}

		echo '<div class="error"><p><strong>' . sprintf( esc_html__( 'WooCommerce PayPal Pro Hosted Disabled', 'woocommerce-gateway-paypal-pro-hosted' ) . '</strong>: ' . esc_html__( 'You must fill the Receiver Email and API Credentials options %s.', 'woocommerce-gateway-paypal-pro-hosted' ), '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=checkout&section=paypal-pro-hosted' ) ) . '">' . esc_html__( 'here', 'woocommerce-gateway-paypal-pro-hosted' ) . '</a>' ) . '</p></div>';
	}

	/**
	 * Adds error message when an unsupported currency is used.
	 *
	 * @return void
	 */
	public function currency_not_supported_message() {
		echo '<div class="error"><p><strong>' . esc_html__( 'WooCommerce PayPal Pro Hosted Disabled', 'woocommerce-gateway-paypal-pro-hosted' ) . '</strong>: ' . sprintf( esc_html__( 'PayPal Pro Hosted does not support your store currency, works only with %s.', 'woocommerce-gateway-paypal-pro-hosted' ), '<code>' . esc_html( implode( ', ', $this->get_supported_currencies() ) ) . '</code>' ) . '</p></div>';
	}

	/**
	 * Adds error message when an unsupported country is used.
	 *
	 * @return void
	 */
	public function country_not_supported_message() {
		echo '<div class="error"><p><strong>' . esc_html__( 'WooCommerce PayPal Pro Hosted Disabled', 'woocommerce-gateway-paypal-pro-hosted' ) . '</strong>: ' . sprintf( esc_html__( 'PayPal Pro Hosted does not support your store country, works only with %s.', 'woocommerce-gateway-paypal-pro-hosted' ), '<code>' . esc_html( implode( ', ', $this->get_supported_countries() ) ) . '</code>' ) . '</p></div>';
	}

	/**
	 * Adds an error message when the api credentials are not valid
	 *
	 * @since 1.0.22
	 * @return void
	 */
	public function api_credentials_not_valid_message() {
		$id = 'woocommerce_' . $this->id;

		// if there are API creds in POST data and don't display any notice yet (this avoids two error notices appearing)
		if ( ! empty( $_POST[ $id . '_api_username' ] ) || ! empty( $_POST[ $id . '_api_password' ] ) || ! empty( $_POST[ $id . '_api_signature' ] ) ) {
			return;
		}

		echo sprintf( esc_html__( '%1$sYou are using invalid PayPal Pro Hosted API credentials.%2$s Please confirm that your PayPal account has been verified and check you have %3$sentered the correct API credentials.%4$s', 'woocommerce-gateway-paypal-pro-hosted' ),
			'<div class="error"><p><strong>',
			'</strong>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=checkout&section=paypal-pro-hosted' ) ) . '">',
			'</a></p></div>'
		);
	}
}
