<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * WC PayPal Pro Hosted API Class.
 */
class WC_Paypal_Pro_Hosted_API {

	/**
	 * Integration version.
	 *
	 * @var string
	 */
	const VERSION = '114.0';

	/**
	 * Gateway class.
	 *
	 * @var WC_Gateway_Paypal_Pro_Hosted
	 */
	protected $gateway;

	/**
	 * Constructor.
	 *
	 * @param WC_Gateway_Paypal_Pro_Hosted $gateway
	 */
	public function __construct( $gateway = null ) {
		$this->gateway    = $gateway;
		$this->notify_url = WC()->api_request_url( 'WC_Gateway_Paypal_Pro_Hosted' );
	}

	/**
	 * Get the API environment.
	 *
	 * @return string
	 */
	protected function get_environment() {
		return ( 'yes' == $this->gateway->get_option( 'sandbox' ) ) ? 'sandbox.' : '';
	}

	/**
	 * Get the NVP URL.
	 *
	 * @return string.
	 */
	public function get_nvp_url() {
		return 'https://api-3t.' . $this->get_environment() . 'paypal.com/nvp';
	}

	/**
	 * Get the payment URL.
	 *
	 * @param  string $hosted_button_id
	 *
	 * @return string.
	 */
	public function get_payment_url( $hosted_button_id ) {
		return 'https://securepayments.' . $this->get_environment() . 'paypal.com/webapps/HostedSoleSolutionApp/webflow/sparta/hostedSoleSolutionProcess?hosted_button_id=' . $hosted_button_id;
	}

	/**
	 * Get the WEBSCR URL.
	 *
	 * @return string.
	 */
	public function get_webscr_url() {
		return 'https://www.' . $this->get_environment() . 'paypal.com/cgi-bin/webscr';
	}

	/**
	 * Do requests.
	 *
	 * @param  string $url      API URL.
	 * @param  string $method   Request method.
	 * @param  array  $data     Request data.
	 * @param  array  $headers  Request headers.
	 *
	 * @return array            Request response.
	 */
	protected function do_request( $url, $method = 'POST', $data = array(), $headers = array() ) {
		$params = array(
			'method'      => $method,
			'timeout'     => 60,
			'user-agent'  => 'WooCommerce/' . WC_VERSION,
			'httpversion' => '1.1',
			'compress'    => false,
			'decompress'  => false,
			'headers'     => array(
				'Content-Type' => 'application/x-www-form-urlencoded'
			)
		);

		if ( 'POST' == $method && ! empty( $data ) ) {
			$params['body'] = $data;
		}

		if ( ! empty( $headers ) ) {
			$params['headers'] = $headers;
		}

		return wp_safe_remote_post( $url, $params );
	}

	/**
	 * Limit the length of item names
	 *
	 * @param  string $item_name
	 *
	 * @return string
	 */
	protected function get_item_name( $item_name ) {
		if ( 127 < strlen( $item_name ) ) {
			$item_name = substr( $item_name, 0, 124 ) . '...';
		}

		return html_entity_decode( $item_name, ENT_NOQUOTES, 'UTF-8' );
	}

	/**
	 * Get the state to send to paypal.
	 *
	 * @param  string $cc
	 * @param  string $state
	 *
	 * @return string
	 */
	protected function get_state( $cc, $state ) {
		if ( 'US' === $cc ) {
			return $state;
		}

		$states = WC()->countries->get_states( $cc );

		if ( isset( $states[ $state ] ) ) {
			return $states[ $state ];
		}

		return $state;
	}

	/**
	 * Get line items to send to paypal
	 *
	 * @param  WC_Order $order
	 *
	 * @return array on success, or false when it is not possible to send line items
	 */
	private function get_line_items( $order ) {
		// Do not send lines for tax inclusive prices
		if ( 'yes' === get_option( 'woocommerce_calc_taxes' ) && 'yes' === get_option( 'woocommerce_prices_include_tax' ) ) {
			return false;
		}

		// Do not send lines when order discount is present, or too many line items in the order.
		if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '2.3', '>=' ) ) {
			$check_order_discount = false;
		} else {
			$check_order_discount = $order->get_order_discount() > 0;
		}

		if ( $check_order_discount || ( sizeof( $order->get_items() ) + sizeof( $order->get_fees() ) ) >= 9 ) {
			return false;
		}

		$item_loop        = 0;
		$args             = array();
		$args['tax_cart'] = $this->price_format( $order->get_total_tax(), $order );

		// Products
		if ( 0 < sizeof( $order->get_items() ) ) {
			foreach ( $order->get_items() as $item ) {
				if ( ! $item['qty'] ) {
					continue;
				}

				$item_loop ++;
				$product   = $order->get_product_from_item( $item );
				$item_name = $item['name'];

				$meta_display = '';
				if ( version_compare( WC_VERSION, '3.0', '<' ) ) {
					$item_meta  = new WC_Order_Item_Meta( $item );
					$meta_display = $item_meta->display( true, true );
					$meta_display = $meta_display ? ( ' ( ' . $meta_display . ' )' ) : '';
				} else {
					foreach ( $item->get_formatted_meta_data() as $meta_key => $meta ) {
						$meta_display .= ' ( ' . $meta->display_key . ':' . $meta->value . ' )';
					}
				}

				if ( $meta_display ) {
					$item_name .= $meta_display;
				}

				$args[ 'item_name_' . $item_loop ] = $this->get_item_name( $item_name );
				$args[ 'quantity_' . $item_loop ]  = $item['qty'];
				$args[ 'amount_' . $item_loop ]    = $this->price_format( $order->get_item_subtotal( $item, false ), $order );

				if ( 0 < $args[ 'amount_' . $item_loop ] ) {
					return false; // Abort - negative line
				}

				if ( $product->get_sku() ) {
					$args[ 'item_number_' . $item_loop ] = $product->get_sku();
				}
			}
		}

		// Discount
		if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '2.3', '>=' ) ) {
			if ( 0 < $order->get_total_discount() ) {
				$args['discount_amount_cart'] = $this->price_format( round( $order->get_total_discount(), 2 ), $order );
			}
		} else {
			if ( 0 < $order->get_cart_discount() ) {
				$args['discount_amount_cart'] = $this->price_format( round( $order->get_cart_discount(), 2 ), $order );
			}
		}

		// Fees
		if ( 0 < sizeof( $order->get_fees() ) ) {
			foreach ( $order->get_fees() as $item ) {
				$item_loop ++;
				$args[ 'item_name_' . $item_loop ] = $this->get_item_name( $item['name'] );
				$args[ 'quantity_' . $item_loop ]  = 1;
				$args[ 'amount_' . $item_loop ]    = $this->price_format( $item['line_total'], $order );

				if ( $args[ 'amount_' . $item_loop ] < 0 ) {
					return false; // Abort - negative line
				}
			}
		}

		// Shipping Cost item - paypal only allows shipping per item, we want to send shipping for the order.
		if ( 0 < $order->get_total_shipping() ) {
			$item_loop ++;
			$args[ 'item_name_' . $item_loop ] = $this->get_item_name( sprintf( __( 'Shipping via %s', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_shipping_method() ) );
			$args[ 'quantity_' . $item_loop ]  = 1;
			$args[ 'amount_' . $item_loop ]    = $this->price_format( $order->get_total_shipping(), $order );
		}

		return $args;
	}

	/**
	 * Get the payment args.
	 *
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	public function get_payment_args( $order ) {
		$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

		// Payment args.
		$args = array(
			'cmd'           => '_hosted-payment',
			'business'      => $this->gateway->receiver_email,
			'no_note'       => 1,
			'currency_code' => get_woocommerce_currency(),
			'charset'       => 'UTF-8',
			'rm'            => is_ssl() ? 2 : 1,
			'upload'        => 1,
			'return'        => add_query_arg( 'utm_nooverride', '1', $this->gateway->get_return_url( $order ) ),
			'cancel_return' => $order->get_cancel_order_url(),
			'paymentaction' => $this->gateway->payment_action,
			'bn'            => 'WooThemes_Cart_ProHSS',

			// Order key + ID.
			'invoice' => $this->gateway->invoice_prefix . ltrim( $order->get_order_number(), '#' ),
			'custom'  => serialize( array( ( $pre_wc_30 ? $order->id : $order->get_id() ), ( $pre_wc_30 ? $order->order_key : $order->get_order_key() ) ) ),

			// Cart.
			'subtotal' => $this->price_format( $order->get_total(), $order ),

			// Template
			'template' => 'templateD',

			// Hosted settings.
			'showHostedThankyouPage' => 'false',
			'showBillingAddress'     => 'false',
			'showShippingAddress'    => 'false',
			'showBillingEmail'       => 'false',
			'showBillingPhone'       => 'false',
			'showCustomerName'       => 'false',
			'showCardInfo'           => 'true',

			// IPN.
			'notify_url' => $this->notify_url,

			// Shipping Address info.
			'first_name' => $pre_wc_30 ? $order->shipping_first_name : $order->get_shipping_first_name(),
			'last_name'  => $pre_wc_30 ? $order->shipping_last_name : $order->get_shipping_last_name(),
			'company'    => $pre_wc_30 ? $order->shipping_company : $order->get_shipping_company(),
			'address1'   => $pre_wc_30 ? $order->shipping_address_1 : $order->get_shipping_address_1(),
			'address2'   => $pre_wc_30 ? $order->shipping_address_2 : $order->get_shipping_address_2(),
			'city'       => $pre_wc_30 ? $order->shipping_city : $order->get_shipping_city(),
			'state'      => $this->get_state( $pre_wc_30 ? $order->shipping_country : $order->get_shipping_country(), $pre_wc_30 ? $order->shipping_state : $order->get_shipping_state() ),
			'zip'        => $pre_wc_30 ? $order->shipping_postcode : $order->get_shipping_postcode(),
			'country'    => $pre_wc_30 ? $order->shipping_country : $order->get_shipping_country(),

			// Billing Address info.
			'billing_first_name' => $pre_wc_30 ? $order->billing_first_name : $order->get_billing_first_name(),
			'billing_last_name'  => $pre_wc_30 ? $order->billing_last_name : $order->get_billing_last_name(),
			'billing_company'    => $pre_wc_30 ? $order->billing_company : $order->get_billing_company(),
			'billing_address1'   => $pre_wc_30 ? $order->billing_address_1 : $order->get_billing_address_1(),
			'billing_address2'   => $pre_wc_30 ? $order->billing_address_2 : $order->get_billing_address_2(),
			'billing_city'       => $pre_wc_30 ? $order->billing_city : $order->get_billing_city(),
			'billing_state'      => $this->get_state( $pre_wc_30 ? $order->billing_country : $order->get_billing_country(), $pre_wc_30 ? $order->billing_state : $order->get_billing_state() ),
			'billing_zip'        => $pre_wc_30 ? $order->billing_postcode : $order->get_billing_postcode(),
			'billing_country'    => $pre_wc_30 ? $order->billing_country : $order->get_billing_country(),
			'email'              => $pre_wc_30 ? $order->billing_email : $order->get_billing_email(),
		);

		$args['buyer_email'] = $args['email'];

		if ( $pre_wc_30 ) {
			$order->billing_phone = str_replace( array( '(', '-', ' ', ')', '.' ), '', $order->billing_phone );
			$billing_phone = $order->billing_phone;
		} else {
			$billing_phone = str_replace( array( '(', '-', ' ', ')', '.' ), '', $order->get_billing_phone() );
			$order->set_billing_phone( $billing_phone );
		}

		// Phone.
		if ( in_array( $args['country'], array( 'US','CA' ) ) ) {

			$args['night_phone_a'] = substr( $billing_phone, 0, 3 );
			$args['night_phone_b'] = substr( $billing_phone, 3, 3 );
			$args['night_phone_c'] = substr( $billing_phone, 6, 4 );
			$args['day_phone_a']   = substr( $billing_phone, 0, 3 );
			$args['day_phone_b']   = substr( $billing_phone, 3, 3 );
			$args['day_phone_c']   = substr( $billing_phone, 6, 4 );
		} else {
			$args['night_phone_b'] = $billing_phone;
			$args['day_phone_b']   = $billing_phone;
		}

		// Try to send line items, or default to sending the order as a whole.
		if ( $line_items = $this->get_line_items( $order ) ) {
			$args = array_merge( $args, $line_items );
		} else {
			$discount = $order->get_total_discount();

			// Discount.
			$args['discount_amount_cart'] = $this->price_format( $discount, $order );

			// Don't pass items - paypal borks tax due to prices including tax. PayPal has no option for tax inclusive pricing sadly. Pass 1 item for the order items overall.
			$item_names = array();

			if ( 0 < sizeof( $order->get_items() ) ) {
				foreach ( $order->get_items() as $item ) {
					if ( $item['qty'] ) {
						$item_names[] = $item['name'] . ' x ' . $item['qty'];
					}
				}
			}

			$args['item_name_1'] = $this->get_item_name( sprintf( __( 'Order %s' , 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_order_number() ) . ' - ' . implode( ', ', $item_names ) );
			$args['quantity_1']  = 1;
			$args['amount_1']    = $this->price_format( $order->get_total() - round( $order->get_total_shipping() + $order->get_shipping_tax(), 2 ) + $discount, $order );

			// Shipping Cost
			// No longer using shipping_1 because
			//      a) paypal ignore it if *any* shipping rules are within paypal
			//      b) paypal ignore anything over 5 digits, so 999.99 is the max
			if ( ( $order->get_total_shipping() + $order->get_shipping_tax() ) > 0 ) {
				$args['item_name_2'] = $this->get_item_name( __( 'Shipping via', 'woocommerce-gateway-paypal-pro-hosted' ) . ' ' . ucwords( $order->get_shipping_method() ) );
				$args['quantity_2']  = 1;
				$args['amount_2']    = $this->price_format( $order->get_total_shipping() + $order->get_shipping_tax(), $order );
			}
		}

		// Mobile template.
		if ( wp_is_mobile() ) {
			$args['template'] = 'mobile-iframe';
		}

		$args = apply_filters( 'woocommerce_paypal_pro_hosted_payment_args', $args );

		return $args;
	}

	/**
	 * Get the button args.
	 *
	 * @param WC_Order $order
	 *
	 * @return array
	 */
	protected function get_button_args( $order ) {

		$args = array(
			'METHOD'     => 'BMCreateButton',
			'BUTTONCODE' => 'TOKEN',
			'BUTTONTYPE' => 'PAYMENT',
			'VERSION'    => self::VERSION,
			'USER'       => $this->gateway->api_username,
			'PWD'        => $this->gateway->api_password,
			'SIGNATURE'  => $this->gateway->api_signature
		);

		$item = 0;
		foreach ( $this->get_payment_args( $order ) as $key => $value ) {
			$args[ 'L_BUTTONVAR' . $item ] = $key . '=' . $value;

			$item++;
		}

		return $args;
	}

	/**
	 * Get the button token ID.
	 *
	 * @param  WC_Order $order
	 *
	 * @return string
	 */
	public function get_button_token_id( $order ) {

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'Generating payment button ID for order ' . $order->get_order_number() . '. Notify URL: ' . $this->notify_url );
		}

		$data     = $this->get_button_args( $order );
		$output   = array();
		$response = $this->do_request( $this->get_nvp_url(), 'POST', $data );

		if ( is_wp_error( $response ) ) {
			if ( 'yes' == $this->gateway->debug ) {
				$this->gateway->log->add( $this->gateway->id, 'WP_Error in generate payment button ID: ' . $response->get_error_message() );
			}
		} elseif ( 200 == $response['response']['code'] ) {
			parse_str( $response['body'], $output );

			if ( isset( $output['ACK'] ) && 'Success' == $output['ACK'] && isset( $output['HOSTEDBUTTONID'] ) ) {
				if ( 'yes' == $this->gateway->debug ) {
					$this->gateway->log->add( $this->gateway->id, 'Payment button ID generated successfully: ' . $output['HOSTEDBUTTONID'] );
				}

				return $output['HOSTEDBUTTONID'];
			} else {
				if ( 'yes' == $this->gateway->debug ) {
					$this->gateway->log->add( $this->gateway->id, 'Failed to generate the payment button ID: ' . print_r( $output, true ) );
				}
			}

		} else {
			if ( 'yes' == $this->gateway->debug ) {
				$this->gateway->log->add( $this->gateway->id, 'Failed to generate a payment button ID: ' . print_r( $response, true ) );
			}
		}

		return '';
	}

	/**
	 * Check PayPal IPN validity.
	 *
	 * @param  array $ipn_response
	 *
	 * @return bool
	 */
	public function check_ipn_request_is_valid( $ipn_response ) {

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'Checking IPN response is valid via ' . $this->get_webscr_url() . '...' );
		}

		// Get received values from post data.
		$validate_ipn = array( 'cmd' => '_notify-validate' );
		$validate_ipn += stripslashes_deep( $ipn_response );

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'IPN Request: ' . print_r( $validate_ipn, true ) );
		}

		// Post back to get a response.
		$response = $this->do_request( $this->get_webscr_url(), 'POST', $validate_ipn );

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'IPN Response: ' . print_r( $response, true ) );
		}

		// Check to see if the request was valid.
		if ( ! is_wp_error( $response ) && $response['response']['code'] >= 200 && $response['response']['code'] < 300 && strstr( $response['body'], 'VERIFIED' ) ) {
			if ( 'yes' == $this->gateway->debug ) {
				$this->gateway->log->add( $this->gateway->id, 'Received valid response from PayPal' );
			}

			return true;
		}

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'Received invalid response from PayPal' );

			if ( is_wp_error( $response ) ) {
				$this->gateway->log->add( $this->gateway->id, 'Error response: ' . $response->get_error_message() );
			}
		}

		return false;
	}

	/**
	 * Get order by custom/invoice data.
	 *
	 * @param  string $custom
	 * @param  string $invoice
	 *
	 * @return WC_Order object
	 */
	public function get_order( $custom, $invoice = '' ) {
		$custom = maybe_unserialize( $custom );

		list( $order_id, $order_key ) = $custom;

		$order = wc_get_order( $order_id );
		$pre_wc_30 = version_compare( WC_VERSION, '3.0', '<' );

		// We have an invalid $order_id, probably because invoice_prefix has changed.
		if ( ! is_object( $order ) ) {
			$order_id = wc_get_order_id_by_order_key( $order_key );
			$order    = wc_get_order( $order_id );
		}

		// Validate key.
		$check_order_key = $pre_wc_30 ? $order->order_key : $order->get_order_key();

		if ( $check_order_key !== $order_key ) {
			if ( 'yes' == $this->gateway->debug ) {
				$this->gateway->log->add( $this->gateway->id, 'Error: Order Key does not match invoice.' );
			}

			exit;
		}

		return $order;
	}

	/**
	 * Do refund.
	 *
	 * @param  WC_Order $order
	 * @param  float    $amount
	 * @param  string   $reason
	 *
	 * @return array
	 */
	public function do_refund( $order, $amount = null, $reason = '' ) {
		$data = array(
			'METHOD'        => 'RefundTransaction',
			'TRANSACTIONID' => $order->get_transaction_id(),
			'REFUNDTYPE'    => is_null( $amount ) ? 'Full' : 'Partial',
			'VERSION'       => self::VERSION,
			'USER'          => $this->gateway->api_username,
			'PWD'           => $this->gateway->api_password,
			'SIGNATURE'     => $this->gateway->api_signature
		);

		if ( ! is_null( $amount ) ) {
			$data['AMT']          = $this->price_format( $amount, $order );
			$data['CURRENCYCODE'] = version_compare( WC_VERSION, '3.0', '<' ) ? $order->get_order_currency() : $order->get_currency();
		}

		if ( $reason ) {
			if ( 255 < strlen( $reason ) ) {
				$reason = substr( $reason, 0, 252 ) . '...';
			}

			$data['NOTE'] = html_entity_decode( $reason, ENT_NOQUOTES, 'UTF-8' );
		}

		return $this->do_request( $this->get_nvp_url(), 'POST', $data );
	}

	/**
	 * Do capture.
	 *
	 * @param  WC_Order $order
	 *
	 * @return array
	 */
	public function do_capture( $order ) {
		$data = array(
			'METHOD'          => 'DoCapture',
			'AUTHORIZATIONID' => $order->get_transaction_id(),
			'AMT'             => $this->price_format( $order->get_total(), $order ),
			'CURRENCYCODE'    => version_compare( WC_VERSION, '3.0', '<' ) ? $order->get_order_currency() : $order->get_currency(),
			'COMPLETETYPE'    => 'Complete',
			'INVNUM'          => $this->gateway->invoice_prefix . ltrim( $order->get_order_number(), '#' ),
			'VERSION'         => self::VERSION,
			'USER'            => $this->gateway->api_username,
			'PWD'             => $this->gateway->api_password,
			'SIGNATURE'       => $this->gateway->api_signature
		);

		return $this->do_request( $this->get_nvp_url(), 'POST', $data );
	}

	/**
	 * Get Payer ID Details
	 *
	 * Developer doc ref: https://developer.paypal.com/docs/archive/nvp-soap-api/merchant/get-paldetails-nvp/
	 *
	 * @return array
	 */
	public function get_pal_details() {
		$data = array(
			'METHOD'    => 'GetPalDetails',
			'VERSION'   => self::VERSION,
			'USER'      => $this->gateway->get_option( 'api_username' ),
			'PWD'       => $this->gateway->get_option( 'api_password' ),
			'SIGNATURE' => $this->gateway->get_option( 'api_signature' ),
		);

		return $this->do_request( $this->get_nvp_url(), 'POST', $data );
	}

	/**
	 * Process the capture.
	 *
	 * @param  WC_Order $order
	 *
	 * @return array
	 */
	public function process_capture( $order ) {
		$response = $this->do_capture( $order );
		$order_id = version_compare( WC_VERSION, '3.0', '<' ) ? $order->id : $order->get_id();

		if ( ! is_wp_error( $response ) && $response['response']['code'] >= 200 && $response['response']['code'] < 300 ) {
			parse_str( $response['body'], $parsed_response );

			if ( 'success' == strtolower( $parsed_response['ACK'] ) ) {
				if ( 'yes' == $this->gateway->debug ) {
					$this->gateway->log->add( $this->gateway->id, 'Captured successfully: ' . print_r( $parsed_response, true ) );
				}

				if ( 'completed' == strtolower( $parsed_response['PAYMENTSTATUS'] ) ) {
					$data = __( 'Transaction captured successfully!', 'woocommerce-gateway-paypal-pro-hosted' );
					$order->add_order_note( __( 'Payment captured on PayPal.', 'woocommerce-gateway-paypal-pro-hosted' ) );
				} else {
					$data = '';

					if ( 'none' != strtolower( $parsed_response['PENDINGREASON'] ) ) {
						$data .= sanitize_text_field( $parsed_response['PENDINGREASON'] ) . '. ';
					}

					if ( 'none' != strtolower( $parsed_response['REASONCODE'] ) ) {
						$data .= sanitize_text_field( $parsed_response['REASONCODE'] ) . '.';
					}

					$order->add_order_note( sprintf( __( 'PayPal: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $data ) );
				}

				update_post_meta( $order_id, '_paypal_pro_hosted_capture_status', 'captured' );

				return array( 'success' => true, 'data' => $data );
			} else {
				if ( 'yes' == $this->gateway->debug ) {
					$this->gateway->log->add( $this->gateway->id, 'Capture failed: ' . print_r( $parsed_response, true ) );
				}

				update_post_meta( $order_id, '_paypal_pro_hosted_capture_status', 'failed' );
				$data = isset( $parsed_response['L_LONGMESSAGE0'] ) ? sanitize_text_field( $parsed_response['L_LONGMESSAGE0'] ) . '.' : '';

				return array( 'success' => false, 'data' => $data );
			}
		}

		if ( 'yes' == $this->gateway->debug ) {
			$this->gateway->log->add( $this->gateway->id, 'Invalid capture: ' . print_r( $response['body'], true ) );
		}

		return array( 'success' => false, 'data' => __( 'An unexpected error occurred while processing the transaction capture. Please turn on logging and try again to see what went wrong.', 'woocommerce-gateway-paypal-pro-hosted' ) );
	}

	/**
	 * Check if currency has decimals.
	 *
	 * @param  string $currency Currency to check.
	 * @return bool
	 */
	public function currency_has_decimals( $currency ) {
		if ( in_array( $currency, array( 'HUF', 'JPY', 'TWD' ), true ) ) {
			return false;
		}
		return true;
	}

	/**
	 * Format prices.
	 *
	 * @param  float|int $price Price to format.
	 * @param  WC_Order  $order Order object.
	 * @return string
	 */
	public function price_format( $price, $order ) {
		$decimals = 2;
		$currency = version_compare( WC_VERSION, '3.0', '<' ) ? $order->get_order_currency() : $order->get_currency();
		if ( ! $this->currency_has_decimals( $currency ) ) {
			$decimals = 0;
		}
		return number_format( $price, $decimals, '.', '' );
	}

	/**
	 * Test the PayPal Pro Hosted API credentials
	 *
	 * @since 1.0.22
	 * @return bool
	 */
	public function test_api_credentials() {
		$api_credentials_valid = false;
		$response              = $this->get_pal_details();

		if ( ! is_wp_error( $response ) && $response['response']['code'] >= 200 && $response['response']['code'] < 300 ) {
			parse_str( $response['body'], $parsed_response );

			// Requests from invalid and unverified PayPal accounts will return an error code 10002
			if ( ! empty( $parsed_response['ACK'] ) && in_array( strtolower( $parsed_response['ACK'] ), array( 'success', 'successwithwarning' ) ) && ! preg_match( '/L_ERRORCODE\d+=10002/', $response['body'] ) ) {
				$api_credentials_valid = true;
			} else if ( 'yes' == $this->gateway->get_option( 'debug' ) ) {
				$this->gateway->log->add( $this->gateway->id, 'API credentials invalid. Response: ' . print_r( $parsed_response, true ) );
			}

		} else if ( 'yes' == $this->gateway->get_option( 'debug' ) ) {
			$this->gateway->log->add( $this->gateway->id, 'Request to test API credentials failed. Response: ' . print_r( $response, true ) );
		}

		set_transient( 'wc_paypal_pro_hosted_' . ( 'yes' == $this->gateway->get_option( 'sandbox' ) ? 'sandbox_' : '' ) . 'account_' . md5( $this->gateway->get_option( 'api_username' ) . ':' . $this->gateway->get_option( 'api_password' ) . ':' . $this->gateway->get_option( 'api_signature' ) ) . '_verified', $api_credentials_valid ? 'yes' : 'no', MONTH_IN_SECONDS );

		return $api_credentials_valid;
	}
}
