<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * WC PayPal Pro Hosted Capture Class.
 */
class WC_Paypal_Pro_Hosted_Capture {

	/**
	 * Constructor.
	 */
	public function __construct() {
		if ( is_admin() ) {
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'add_meta_boxes', array( $this, 'capture_meta_box' ) );
		}

		add_action( 'wp_ajax_wc_paypal_pro_hosted_capture', array( $this, 'ajax_capture' ) );
	}

	/**
	 * Enqueue scripts.
	 */
	public function enqueue_scripts() {
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_register_script( 'wc-paypal-pro-hosted-capture', plugins_url( 'assets/js/capture' . $suffix . '.js', plugin_dir_path( __FILE__ ) ), array( 'jquery' ), WC_PAYPAL_PRO_HOSTED_VERSION, true );
		wp_localize_script(
			'wc-paypal-pro-hosted-capture',
			'WCPayPalProHostedCaptureParams',
			array(
				'security' => wp_create_nonce( 'wc_paypal_pro_hosted_capture_nonce' ),
			)
		);
	}

	/**
	 * Order capture meta box.
	 */
	public function capture_meta_box() {
		$screen = get_current_screen();

		if ( isset( $screen->id ) && 'shop_order' == $screen->id ) {
			$settings = get_option( 'woocommerce_paypal-pro-hosted_settings', array() );

			if ( isset( $settings['payment_action'] ) && 'authorization' == $settings['payment_action'] ) {
				add_meta_box(
					'paypal-hosted-pro-capture',
					__( 'PayPal Pro Hosted', 'woocommerce-gateway-paypal-pro-hosted' ),
					array( $this, 'capture_meta_box_content' ),
					'shop_order',
					'side',
					'low',
					$settings
				);
			}
		}
	}

	/**
	 * Orde capture meta box content.
	 *
	 * @param WP_Post $post
	 * @param array $meta
	 */
	public function capture_meta_box_content( $post, $meta ) {
		$order = new WC_Order( $post->ID );

		$check_payment_method = version_compare( WC_VERSION, '3.0', '<' ) ? $order->payment_method : $order->get_payment_method();
		if ( 'paypal-pro-hosted' !== $check_payment_method ) {
			echo "<script>document.getElementById( 'paypal-hosted-pro-capture' ).style.display = 'none';</script>";
			return;
		}

		if ( ! $meta['args']['api_username'] || ! $meta['args']['api_password'] || ! $meta['args']['api_signature'] ) {
			echo '<div class="error" style="margin: 0 !important;"><p>' . __( 'You must fill your API key in order to capture payments!', 'woocommerce-gateway-paypal-pro-hosted' ) . '</p></div>';
			return;
		}

		if ( ! $order->get_transaction_id() ) {
			echo '<p>' . __( 'The transaction was not authorized yet.', 'woocommerce-gateway-paypal-pro-hosted' ) . '</p>';
			return;
		}

		if ( 'captured' === get_post_meta( $post->ID, '_paypal_pro_hosted_capture_status', true ) ) {
			echo '<div class="updated" style="margin: 0 !important;"><p>' . __( 'This transaction has already been captured.', 'woocommerce-gateway-paypal-pro-hosted' ) . '</p></div>';
			return;
		}

		wp_enqueue_script( 'wc-paypal-pro-hosted-capture' );

		echo '<button type="button" class="button-primary" id="paypal-pro-hosted-capture" data-order_id="' . ( version_compare( WC_VERSION, '3.0', '<' ) ? $order->id : $order->get_id() ) . '">' . sprintf( __( 'Capture: %s', 'woocommerce-gateway-paypal-pro-hosted' ), $order->get_formatted_order_total() ) . '</button>';
	}

	/**
	 * Capture order by ajax.
	 */
	public function ajax_capture() {
		ob_start();

		check_ajax_referer( 'wc_paypal_pro_hosted_capture_nonce', 'security' );

		$order_id = wc_clean( stripslashes( $_POST['order_id'] ) );

		if ( empty( $order_id ) ) {
			die();
		}

		$order            = new WC_Order( $order_id );
		$payment_gateways = WC()->payment_gateways->payment_gateways();
		$api              = $payment_gateways['paypal-pro-hosted']->api;
		$response         = $api->process_capture( $order );

		wp_send_json( $response );
	}
}

new WC_Paypal_Pro_Hosted_Capture();
